#import "ViewController.h"
#import <UIKit/UIKit.h>
#import "EaglUIVIew.h"
#import "VSLogging.h"
#import "GStreamerDisplayBackend.h"

@interface ViewController () {
    int media_width;
    int media_height;
    
    CGFloat nextX;
    CGFloat nextY;
    
    NSMutableArray *backends;
    GStreamerDisplayBackend *_transformingDisplayBackend;
}

@end

@implementation ViewController

/*
 * Methods from UIViewController
 */

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    /* Make these constant for now, later tutorials will change them */
    media_width = 640;
    media_height = 480;
    
    nextX = 0;
    nextY = 0;
    
    backends = [[NSMutableArray alloc] init];
    
    //
    // VIDEO DISPLAYS
    //
    
    int displayCount = 3;
    for (int i = 0; i < displayCount; i++)
    {
        [self addVideoView:NO];
    }
    
    // get gstreamer version using first backend available
    if (backends.count > 0)
    {
        GStreamerBackend *backend = [backends objectAtIndex:0];
        [message_label setText: [backend getGStreamerVersion]];
    }
    
    // call after gst_backend is created
    [self updateControls];
}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

/* Called when the Play button is pressed */
-(IBAction) play:(id)sender
{
    for (GStreamerBackend *backend in backends)
    {
        [backend start];
    }
    play_button.enabled = NO;
}

/* Called when the Pause button is pressed */
-(IBAction) stop:(id)sender
{
    for (GStreamerBackend *backend in backends)
    {
        [backend stop];
    }
    pause_button.enabled = NO;
}

-(void) addVideoView:(BOOL)doStart
{
    int vidCount = backends.count;
    
    CGFloat x, y;
    
    x = nextX;
    y = nextY;
    
    EaglUIView *newVideoView = [[EaglUIView alloc] initWithFrame:CGRectMake(x, y, media_width, media_height)];
    [self.view addSubview:newVideoView];
    
    nextX += 100;
    nextY += 100;
    
    NSString *streamId = [NSString stringWithFormat:@"display%u", (vidCount + 1)];
    GStreamerDisplayBackend *newVideoDisplay = [[GStreamerDisplayBackend alloc] initWithDelegate:self streamId:streamId displayView:newVideoView];

    // attach pinch gesture recognizer
    UIPinchGestureRecognizer *pinchRecognizer = [[UIPinchGestureRecognizer alloc] initWithTarget:self action:@selector(handlePinchGesture:)];
    [newVideoDisplay.displayView addGestureRecognizer:pinchRecognizer];
    
    [backends addObject:newVideoDisplay];
    
    if (doStart)
        [newVideoDisplay start];
}

// track initial point when video begins move
- (void) touchesBegan:(NSSet *)touches withEvent:(UIEvent *)event
{
    _transformingDisplayBackend = nil;
    for (UITouch *touch in touches)
    {
        for (GStreamerDisplayBackend *backend in backends)
        {
            if (touch.view == backend.displayView)
            {
                backend.firstTouchPosition = [touch locationInView:backend.displayView.superview];
                backend.displayCenterAtFirstTouch = backend.displayView.center;
                [backend.displayView.superview bringSubviewToFront:backend.displayView];
                _transformingDisplayBackend = backend;
                break;
            }
        }
    }
}

// update video position as touches move
- (void) touchesMoved:(NSSet *)touches withEvent:(UIEvent *)event
{
    for (UITouch *touch in touches)
    {
        GStreamerDisplayBackend *backend = _transformingDisplayBackend;
        if (touch.view == backend.displayView)
        {
            CGPoint position = [touch locationInView:backend.displayView.superview];
            
            CGFloat dX = backend.firstTouchPosition.x - position.x;
            CGFloat dY = backend.firstTouchPosition.y - position.y;
            
            backend.displayView.center = CGPointMake(backend.displayCenterAtFirstTouch.x - dX, backend.displayCenterAtFirstTouch.y - dY);
            
            break;
        }
    }
}

/**
 * Callback for pinch handling
 */
- (void)handlePinchGesture:(UIPinchGestureRecognizer *)recognizer
{
    if (recognizer.state == UIGestureRecognizerStateBegan)
    {
        LOG_DEBUG(@"BEGIN pinch gesture!");
        _transformingDisplayBackend = nil;
        for (GStreamerDisplayBackend *backend in backends)
        {
            if (backend.displayView == recognizer.view)
            {
                _transformingDisplayBackend = backend;
                break;
            }
        }
        if (_transformingDisplayBackend != nil)
        {
            dispatch_async(dispatch_get_main_queue(), ^{
                [_transformingDisplayBackend stop];
            });
        }
    }
    else if (recognizer.state == UIGestureRecognizerStateEnded)
    {
        CGFloat scale = recognizer.scale;
        LOG_DEBUG(@"END pinch gesture! scale = %0.2f", scale);
        
        dispatch_async(dispatch_get_main_queue(), ^{
            [_transformingDisplayBackend start];
        });
    }
    else
    {
        CGFloat scale = recognizer.scale;
        LOG_DEBUG(@"pinch gesture! scale = %0.2f", scale);
        
        if (scale > 1)
        {
            scale = 1 + ((scale - 1) / 2);
        }
        else if (scale < 1)
        {
            scale = 1 - ((1 - scale) / 2);
        }
        
        if (scale != 1)
        {
            __weak UIView *displayView = _transformingDisplayBackend.displayView;
            
            CGRect displayFrame = displayView.frame;
            
            NSUInteger curWidth = (NSUInteger)displayFrame.size.width;
            NSUInteger curHeight = (NSUInteger)displayFrame.size.height;
            
            NSUInteger newWidth = (NSUInteger)curWidth * scale;
            NSUInteger newHeight = (NSUInteger)curHeight * scale;
            
            NSInteger dWidth = curWidth - newWidth;
            NSInteger dHeight = curHeight - newHeight;
            
            // impose minimum width and height restrictions
            if (newWidth < 320 || newHeight < 240)
                return;
            
            // impose maximum width and height restrictions
            UIView *superView = displayView.superview;
            if (superView != nil)
            {
                if (newWidth > superView.frame.size.width || newHeight > superView.frame.size.height)
                    return;
            }
            
            NSInteger dWidthHalf = (NSInteger)dWidth / 2;
            NSInteger dHeightHalf = (NSInteger)dHeight / 2;
            
            dispatch_async(dispatch_get_main_queue(), ^{
                displayView.frame = CGRectMake(displayFrame.origin.x + dWidthHalf, displayFrame.origin.y + dHeightHalf, newWidth, newHeight);
            });
        }
    }
}

#pragma mark VideoStreamBackendDelegate methods

/*
 * Methods from GstreamerBackendDelegate
 */

-(void) updateControls
{
    BOOL isActive = YES;
    for (GStreamerBackend *backend in backends)
    {
        isActive &= backend.isActive;
    }
    play_button.enabled = !isActive;
    pause_button.enabled = isActive;
}

-(void) streamRequestFailed:(id)sender reason:(NSString *)reason
{
    dispatch_async(dispatch_get_main_queue(), ^{
        message_label.text = reason;
        LOG_DEBUG(@"failed to ready stream");
        [self updateControls];
    });
}

-(void) streamActiveChanged:(id)sender isActive:(BOOL)isActive
{
    dispatch_async(dispatch_get_main_queue(), ^{
        LOG_DEBUG(@"active changed to: %hhd", isActive);
        [self updateControls];
    });
}

-(void) streamMessage:(id)sender message:(NSString *)message
{
    dispatch_async(dispatch_get_main_queue(), ^{
        message_label.text = message;
    });
}

@end
